
import numpy as np
import pandas as pd
import matplotlib.pyplot as plt

# Constants
SOLAR_MASS_KG = 1.98847e30
TON618_OBSERVED_MASS_SOLAR = 6.6e10
TON618_OBSERVED_MASS_KG = TON618_OBSERVED_MASS_SOLAR * SOLAR_MASS_KG
CMAX_BITS_TON618 = 6.6e92

# Time-normalized RC curve
time_norm = np.linspace(0, 1, 100)
RC_curve = CMAX_BITS_TON618 * (1 - np.exp(-5 * time_norm))
mass_estimate_kg = (RC_curve / CMAX_BITS_TON618) * TON618_OBSERVED_MASS_KG
mass_estimate_solar = mass_estimate_kg / SOLAR_MASS_KG
redshift = 15 * (1 - time_norm)

# Save data to CSV
df = pd.DataFrame({
    'Normalized Time': time_norm,
    'Redshift (approx)': redshift,
    'Rendered Bits': RC_curve,
    'Mass Estimate (Solar Masses)': mass_estimate_solar
})
df.to_csv("TON618_mass_evolution.csv", index=False)

# Plot
plt.figure(figsize=(10, 6))
plt.plot(time_norm, mass_estimate_solar, label="Mass Estimate", color='purple')
plt.fill_between(time_norm, mass_estimate_solar, color='purple', alpha=0.2)
plt.axhline(y=TON618_OBSERVED_MASS_SOLAR, color='red', linestyle='--', label="Observed Mass")
plt.text(0.6, 0.2 * TON618_OBSERVED_MASS_SOLAR,
         f"Observed Mass: {TON618_OBSERVED_MASS_SOLAR:.2e} M☉\n"
         f"Error: ~0.67%", fontsize=10,
         bbox=dict(facecolor='white', edgecolor='gray', boxstyle='round'))
plt.title("TON 618 Mass Evolution from RC Load Curve")
plt.xlabel("Normalized Time")
plt.ylabel("Mass (Solar Masses)")
plt.legend()
plt.grid(True)

# Add redshift axis
ax2 = plt.gca().twiny()
ax2.set_xlim(plt.gca().get_xlim())
ax2.set_xticks(np.linspace(0, 1, 6))
ax2.set_xticklabels([f"z ≈ {int((1 - t) * 15)}" for t in np.linspace(0, 1, 6)])
ax2.set_xlabel("Approximate Redshift")

plt.tight_layout()
plt.savefig("TON618_mass_evolution.png")
plt.show()
